<?php
/* --------------------------------------------------------------
   TrackingCodeRepository.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\Repository;

use Doctrine\DBAL\ConnectionException;
use Gambio\Admin\TrackingCode\Events\AllTrackingCodesRequested;
use Gambio\Admin\TrackingCode\Events\DeletedMultipleTrackingCodes;
use Gambio\Admin\TrackingCode\Events\DeletedSingleTrackingCode;
use Gambio\Admin\TrackingCode\Events\DeletionOfMultipleTrackingCodesRequested;
use Gambio\Admin\TrackingCode\Events\DeletionOfSingleTrackingCodeRequested;
use Gambio\Admin\TrackingCode\Events\FetchedAllTrackingCodes;
use Gambio\Admin\TrackingCode\Events\FetchedSpecificTrackingCode;
use Gambio\Admin\TrackingCode\Events\SpecificTrackingCodeRequested;
use Gambio\Admin\TrackingCode\Events\StorageOfMultipleTrackingCodesRequested;
use Gambio\Admin\TrackingCode\Events\StorageOfSingleTrackingCodeRequested;
use Gambio\Admin\TrackingCode\Events\StoredMultipleTrackingCodes;
use Gambio\Admin\TrackingCode\Events\StoredMultipleTrackingCodesE;
use Gambio\Admin\TrackingCode\Events\StoredSingleTrackingCode;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCode;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeId;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeIds;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodes;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeSqlCriteria;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeSqlPagination;
use Gambio\Admin\TrackingCode\Model\TrackingCodeId as TrackingCodeIdModel;
use Gambio\Admin\TrackingCode\Model\TrackingCodes as TrackingCodesModel;
use InvalidArgumentException;
use Psr\EventDispatcher\EventDispatcherInterface;

/**
 * Class TrackingCodeRepository
 *
 * @package Gambio\Admin\TrackingCode\Repository
 */
class TrackingCodeRepository
{
    /**
     * @var TrackingCodeMapper
     */
    private $mapper;
    
    /**
     * @var TrackingCodeReader
     */
    private $reader;
    
    /**
     * @var TrackingCodeWriter
     */
    private $writer;
    
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;
    
    
    /**
     * TrackingCodeRepository constructor.
     *
     * @param TrackingCodeMapper       $mapper
     * @param TrackingCodeReader       $reader
     * @param TrackingCodeWriter       $writer
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        TrackingCodeMapper $mapper,
        TrackingCodeReader $reader,
        TrackingCodeWriter $writer,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->mapper          = $mapper;
        $this->reader          = $reader;
        $this->writer          = $writer;
        $this->eventDispatcher = $eventDispatcher;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getAll(
        TrackingCodeSqlCriteria $criteria,
        TrackingCodeSqlPagination $pagination
    ): TrackingCodes {
        $event = AllTrackingCodesRequested::create($criteria, $pagination);
        $this->eventDispatcher->dispatch($event);
        
        $trackingCodes     = [];
        $trackingCodesData = $this->reader->getAllData($criteria, $pagination);
        foreach ($trackingCodesData as $trackingCodeData) {
            $trackingCodes[] = $this->mapper->mapTrackingCode($trackingCodeData);
        }
        
        $event = FetchedAllTrackingCodes::create(TrackingCodesModel::create(...$trackingCodes));
        $this->eventDispatcher->dispatch($event);
        
        return $event->trackingCodes();
    }
    
    
    /**
     * @inheritDoc
     */
    public function getTotalCount(TrackingCodeSqlCriteria $criteria): int
    {
        return $this->reader->getTotalCount($criteria);
    }
    
    
    /**
     * @inheritDoc
     */
    public function getById(TrackingCodeId $id): TrackingCode
    {
        $event = SpecificTrackingCodeRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $trackingCode = $this->reader->getById($id);
        
        $event = FetchedSpecificTrackingCode::create($this->mapper->mapTrackingCode($trackingCode));
        $this->eventDispatcher->dispatch($event);
        
        return $event->trackingCode();
    }
    
    
    /**
     * @inheritDoc
     */
    public function store(TrackingCode $trackingCode): TrackingCodeId
    {
        $event = StorageOfSingleTrackingCodeRequested::create($trackingCode);
        $this->eventDispatcher->dispatch($event);
        
        if ($trackingCode->id() !== null) {
            $this->writer->update($trackingCode);
            
            $event = StoredSingleTrackingCode::create(TrackingCodeIdModel::create($trackingCode->id()));
            $this->eventDispatcher->dispatch($event);
            
            return $event->trackingCodeId();
        }
        
        $id = $this->writer->insert($trackingCode);
        
        $event = StoredSingleTrackingCode::create(TrackingCodeIdModel::create($id));
        $this->eventDispatcher->dispatch($event);
        
        return $event->trackingCodeId();
    }
    
    
    /**
     * @inheritDoc
     *
     * @throws ConnectionException
     */
    public function storeMultiple(TrackingCodes $trackingCodes): TrackingCodeIds
    {
        $ids       = [];
        $operation = null;
        
        /** @var $trackingCode TrackingCode $trackingCode */
        foreach ($trackingCodes as $trackingCode) {
            if ($trackingCode->id() !== null) {
                $ids[]     = $trackingCode->id();
                $operation = 'update';
            } elseif ($operation === 'update') {
                throw new InvalidArgumentException('All provided parcel services need to be new or existing.'
                                                   . 'Can not mix store operation for new and existing parcel services.');
            } else {
                $operation = 'insert';
            }
        }
        
        $event = StorageOfMultipleTrackingCodesRequested::create($trackingCodes);
        $this->eventDispatcher->dispatch($event);
        
        if ($operation === 'update') {
            $this->writer->updateMultiple($trackingCodes);
            
            $event = StoredMultipleTrackingCodes::create($this->mapper->mapTrackingCodeIds($ids));
            $this->eventDispatcher->dispatch($event);
            
            return $event->trackingCodeIds();
        }
        
        $ids = $this->writer->insertMultiple($trackingCodes);
        
        $event = StoredMultipleTrackingCodes::create($this->mapper->mapTrackingCodeIds($ids));
        $this->eventDispatcher->dispatch($event);
        
        return $event->trackingCodeIds();
    }
    
    
    /**
     * @inheritDoc
     */
    public function delete(TrackingCodeId $id): void
    {
        $event = DeletionOfSingleTrackingCodeRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->delete($id);
        
        $event = DeletedSingleTrackingCode::create($id);
        $this->eventDispatcher->dispatch($event);
    }
    
    
    /**
     * @inheritDoc
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(TrackingCodeIds $ids): void
    {
        $event = DeletionOfMultipleTrackingCodesRequested::create($ids);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->deleteMultiple($ids);
        
        $event = DeletedMultipleTrackingCodes::create($ids);
        $this->eventDispatcher->dispatch($event);
    }
}